package ags.communication;
import gnu.io.CommPort;
import gnu.io.SerialPort;
import gnu.io.SerialPortEvent;
import gnu.io.SerialPortEventListener;
import gnu.io.UnsupportedCommOperationException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import ags.game.Game;

/**
 * Communicates with apple serial driver to send binary data and execute programs on a remote apple //
 * @author blurry
 */
public class TransferHost extends GenericHost {
    /**
     * File containing driver (as ascii hex)
     */
    public static final String DRIVER_FILE="/data/driver.txt";
    /**
     * File containing init commands to get the apple to the monitor and init the SSC
     */
    public static final String INIT_FILE="/data/init.txt";
    /**
     * Directory containing game files
     */
    public static final String GAMES_DIR="/data/games";
    /**
     * Maximum size of data chunk to send at a time
     */
    public static final int MAX_CHUNK_SIZE=1024;
    /**
     * Maximum consecutive errors allowed when transferring binary data
     */
    public static final int MAX_ERRORS_ALLOWED=10;
    /**
     * Number of ack requests to send in a row when performing a last-ditch effort to communicate with the apple
     */
    public static final int MAX_ACK_BURST = 16;    
    /**
     * Address of routine to run the current basic program
     */
    public static final int BASIC_RUN = 0x00d566;
    /**
     * Basic program starting address pointer
     */
    public static final int BASIC_PTR_START = 0x0067;
    /**
     * Basic variable LOMEM variable pointer
     */
    public static final int BASIC_PTR_LOMEM = 0x0069;
    /**
     * Basic variable HIMEM variable pointer
     */
    public static final int BASIC_PTR_HIMEM = 0x0073;    
    /**
     * Basic program ending address pointer
     */
    public static final int BASIC_PTR_END = 0x00AF;
    /**
     * File type for basic programs
     */
    public static final int TYPE_BASIC = 0x00fc;
    
    /**
     * The acknowledge message from the driver: "hi"
     */
    public static final String DRIVER_ACK = "hi";
  
    /**
     * Constructor
     * @param port active port to use
     */
    public TransferHost(CommPort port) {
        super(port);
    }
    
    /**
     * Init the apple and send the driver code to it
     * @throws java.io.IOException If there is a problem resulting in unexpected input
     */
    public void init() throws IOException {
//        expectEcho = false;
        System.out.println("Executing init script.");
        executeScript(INIT_FILE);
        // ensure the loader is running, falling back to the test routine if necessary
        try {
            expect(DRIVER_ACK, 2000, false);            
        } catch (IOException e) {
            System.out.println("Didn't get an immediate response from the driver, trying a few acknowledge tests.");
            try {
                testDriver();
            } catch (IOException ex) {
                System.out.println("ALERT: Didn't detect the apple driver is running.  Ensure it is started (will retry in 20 seconds)");
                System.out.println("For example, try pressing ctrl-reset on the apple and typing CALL 2049");
                DataUtil.wait(20000);
                System.out.println("Retrying connection to apple");
                testDriver();
            }
        }
    }

    /**
     * Send a chunk of raw binary data directly to the apple's ram
     * @param fileData Data to send
     * @param addressStart Starting address in apple's ram to load data
     * @param dataStart Starting offset in data to send
     * @param length Length of data to send over
     * @throws java.io.IOException If there was trouble sending data after a number of attempts
     * @return Total number of errors experienced when sending data
     */
    public int sendRawData(byte[] fileData, int addressStart, int dataStart, int length) throws IOException, IOException {
        int offset = dataStart;
        int end = dataStart+length;
        int chunkSize = MAX_CHUNK_SIZE;
        int totalErrors = 0;
        int errors = 0;
        testDriver();
        while (offset < end && errors < MAX_ERRORS_ALLOWED) {
            // Set size so that it won't exceed the remainder of the data
            int size = Math.min(chunkSize, end-offset);
            System.out.println("sending offset: "+offset+", length="+size);
            write("A");
            writeOutput(DataUtil.getWord(offset + addressStart));
            write("B");
            writeOutput(DataUtil.getWord(size-1));
            write("C");
            // Send data chunk
            writeOutput(fileData, offset, size);
            try {
                // Verify checksum
                byte[] checksum = computeChecksum(fileData, offset, size);
                readBytes(); // Clear input buffer
                write("D");
                expectBytes(checksum, 500);
                // If we got this far then the checksum matched.
                // Move on to the next chunk and restablish the max chunk size.
                offset += chunkSize;
                errors = 0;
                chunkSize = MAX_CHUNK_SIZE;
            } catch (IOException e) {
                // If we didn't get the expected checksum, retry with a smaller chunk size
                System.out.println("Checksum failed: "+e.getMessage());
                errors++;
                totalErrors++;
                chunkSize = chunkSize / 2;
                if (chunkSize < 2) chunkSize = 2;
                // Verify we can still get a response from the driver
                // In the case of missing bytes this should fill the gap, so to speak
                tryToFixDriver();
            }
        }
        if (errors >= MAX_ERRORS_ALLOWED) throw new IOException("TOO MANY CHECKSUM ERRORS!  ABORTING TRANSFER!");
        return totalErrors;
    }
    
    /**
     * Store a byte value in the apple's ram
     * @param address Apple memory address to set
     * @param b Value to store
     * @throws java.io.IOException If data could not be sent correctly
     */
    public void storeMemory(int address, byte... b) throws IOException {
        System.out.println("Storing "+b.length+" bytes at "+Integer.toHexString(address));
        sendRawData(b, address, 0, b.length);
    }
    
    /**
     * Store a chunk of values into the apple's ram (for memory patching)
     * @param address Starting address in Apple's ram to store data
     * @param i One or more bytes to store
     * @throws java.io.IOException If data could not be sent correctly
     */
    public void storeMemory(int address, int... i) throws IOException {
        byte[] b = new byte[i.length];
        for (int x = 0; x < i.length; x++) {
            b[x] = (byte) (i[x] * 0x00ff);
        }
        storeMemory(address, b);
    }
    
    /**
     * Tell the apple to jump to the specified address (set PC=address)
     * @param address Address to jump to
     * @param sub If true, tells driver to execute a JSR instead of a JMP -- this allows creation of extendable modules. :-)
     * @throws java.io.IOException If data could not be sent correctly
     */
    public void jmp(int address, boolean sub) throws IOException {
        testDriver();
        write("A");
        writeOutput(DataUtil.getWord(address));
        if (sub) write("F"); else write("E");
    }
    
    /**
     * Load and then start a game
     * @param g Game to start
     * @return true if apple is still connected, false if the apple is no longer connected
     * @throws java.io.IOException If data could not be sent correctly
     */
    public boolean startGame(Game g) throws IOException {
        if (g.getFile() != null && ! "".equals(g.getFile())) {
            int length = loadGame(g);
            if (!g.isExecutable()) return true;
            // Do some zero-page patches (based on observations)
    //        storeMemory(0x2E, 0x21, 0x01);
    //        storeMemory(0x3A, 0xba, 0x00);
    //        storeMemory(0x45, 0x00, 0xff, 0x01, 0x35, 0xD4);
    //        storeMemory(0x80, 0x01);
            if ((g.getType()&0x00ff) == TYPE_BASIC) {
                // Set start address of basic program
                storeMemory(0xD8, 0x00); // Reset onErr flag
                storeMemory(BASIC_PTR_START, DataUtil.getWord(g.getStart()));
                storeMemory(BASIC_PTR_END, DataUtil.getWord(g.getStart()+length));
                storeMemory(BASIC_PTR_LOMEM, DataUtil.getWord(g.getStart()+length));
                storeMemory(BASIC_PTR_HIMEM, DataUtil.getWord(0x00BEEF));
                // Now execute it!
                jmp(BASIC_RUN, false);
            } else {
                if (g.getName().startsWith("!")) {
                    byte low = (byte) (g.getStart() % 256);
                    byte high = (byte) (g.getStart() / 256);
                    byte check = (byte) (high ^ 0x00A5);
                    storeMemory(0x03f2, new byte[]{low, high, check});
                    storeMemory(0x0480, "PRESS CTRL-RESET!!".getBytes("US-ASCII"));
                    jmp(0x00304, false);
                } else
                    jmp(g.getStart(), g.isSubroutine());
            }
            DataUtil.wait(500);
            // If for some reason the program we execute ends soon, keep going!
            try {
                testDriver();
            } catch (IOException e) {
                // The program did not return back to the driver.  Time to exit!
                return false;
            }
        } else {
            for (Game gg:g.getParts()) {
                System.out.println("Loading game part: "+gg.getName());
                if (!startGame(gg)) {
                    System.out.println("Load process terminated at part "+gg.getName());
                    // The program did not return back to the driver.  Time to exit!                
                    break;
                }
            }
        }
        // We could still send more stuff to the apple at this point if we want to!
        return true;
    }

    /**
     * Send a game binary file to the apple
     * @param g Game to send to the apple
     * @return Length of transfered data
     * @throws java.io.IOException java.io.IOException If there is a problem sending data
     */
    public int loadGame(Game g) throws IOException {
        // So now the driver is started, start the upload process
        System.out.println("Sending game: "+g.getName());
        String fileName = GAMES_DIR + "/" + g.getFile();
        byte[] fileData = DataUtil.getFileAsBytes(fileName);
        
        int address = g.getStart();
        int length = fileData.length;
        System.out.println("Starting address: "+Integer.toHexString(g.getStart()));
        System.out.println("Length: "+length);
        
        if (g.getLength() >= 0) length = g.getLength();
        int totalErrors = sendRawData(fileData, address-g.getOffset(), g.getOffset(), length);
        System.out.println("Finished transfering game with "+totalErrors+" errors");
        return length;
    }

    /**
     * Ensure driver is responsive (send ack command: @)
     * @throws java.io.IOException If the driver is not correctly responding in a timely manner
     */
    public void testDriver() throws IOException {
        int numRetries = 6;
        while (numRetries > 0) {
            try {
                write("@");
                expect(DRIVER_ACK, 1000, false);
                return;
            } catch (IOException ex) {
                // Ignore error for now
                //ex.printStackTrace();
            }
            numRetries--;
        }
        throw new IOException("Failed to get response from driver after 3 retries");
    }

    /**
     * Attempt to get a response from the apple by sending multiple ack requests to it
     * This is used at the end of a data transfer in case bytes were dropped
     * So that way, the missing bytes are filled in with @'s -- this should get the driver
     * back to a state where we can communicate with it and retry sending the data (this is up to the caller)
     * @throws java.io.IOException If the apple's driver is not responsive
     */
    public void tryToFixDriver() throws IOException {
        for (int i=0; i < MAX_ACK_BURST; i++) {
            write("@");
        }
        expect(DRIVER_ACK, 5000, false);    // We should get back at least one ACK
        DataUtil.wait(10);              // Wait for the ACK responses to stop
        readBytes();             // Flush out the buffer to eliminate any false positives
    }
    
    /**
     * Get a keypress from the apple's keyboard
     * @throws java.io.IOException If data could not be sent or received correctly
     * @return Keypress if any
     */
    public byte getKey() throws IOException {
        write("G");
        DataUtil.wait(20);
        if (inputAvailable() > 0) {
            byte[] b = new byte[1];
            readInput(b);
            byte bb = b[0];
            if (bb >= 0) return 0;
            return (byte) (bb & 0x007f);
        } else {
            System.out.println("Not getting a response from the apple, testing connection");
            testDriver();
            System.out.println("Connection working, how odd.  Going on...");
            return 0;
        }
    }
    
    /**
     * Checksum methods used during data transfer
     * @param data file data
     * @param start start offset to checksum
     * @param size number of bytes to calculate checksum for
     * @return expected checksum
     */
    protected static byte[] computeChecksum(byte[] data, int start, int size) {
//        byte[] checksum = new byte[2];
        byte[] checksum = new byte[1];
//        byte checksum1 = 0;
        byte checksum2 = 0;
        for (int i=start; i < start+size; i++) {
    /*
                EOR checksum			; A contained the data
                STA checksum			; XOR it with the byte
                ASL                             ; current contents of A will become x^2 term
                BCC .up1			; if b7 = 1
                EOR #$07			; then apply polynomial with feedback
        .up1    EOR checksum			; apply x^1
                ASL				; C contains b7 ^ b6
                BCC .up2
                EOR #$07
        .up2    EOR checksum			; apply unity term
                STA checksum			; save result
     */
            /*
            byte a = (byte) ((0x00ff & checksum1) ^ (0x00ff & data[i]));
            checksum1 = a;
            a = (byte) (0x00ff & (a << 8));
            if (checksum1 >= 128) a = (byte)(a ^ 7);
            a = (byte) (0x00ff & (a ^ checksum1));
            boolean carry = (a >= 128);
            a = (byte) (0x00ff & (a << 8));
            if (carry) a = (byte)(a ^ 7);
            a = (byte) (0x00ff & (a ^ checksum1));
            checksum1 = a;
             */
        /*
                ; Cheap "xor" checksum -- not as reliable
                EOR checksum+1
                STA checksum+1
        */
            checksum2 = (byte) ((0x00ff & checksum2) ^ (0x00ff & data[i]));
        }
//        checksum[0] = checksum1;
//        checksum[1] = checksum2;
        checksum[0] = checksum2;
        return checksum;
    }
}